/**
* \file: errmemd_lib.h
*
* Definition of the socket interface to connect to errmem backend.
* 
* This file defines the socket interface to establish sessions
* with the error memoy backend (daemon) and to use the backend's
* functionality.
*
* \component: errmemd_lib
*
* \author: Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#ifndef _ERRMEMD_LIB_H
#define _ERRMEMD_LIB_H

#include <stdio.h>
#include <stdint.h>

#include <linux/errmem.h>

/* Not persistent output devices supported by the daemon
 * to directly dump messages to
 */
#define ERRMEM_DLT_OUT   0x00010000  /* output to DLT interface   - opt: -g   */
#define ERRMEM_STD_OUT   0x00020000  /* output to standard out                */
#define ERRMEM_STD_ERR   0x00040000  /* output to standard error              */
#define ERRMEM_ALL_OUT   0x000F0000  /* mask containing all output devices    */

/* Dumping messages from the frontend to an output device
 * can be a one-time job, means "once" or continuously */
#define ERRMEM_DUMP_CONT 0x00000000  /* dump is continuously */
#define ERRMEM_DUMP_ONCE 0x00000010  /* dump til no (new) message is available */

/* Dumping messages from the frontend to an output device
 * can imply the acknowledging of messages to the frontend */
#define ERRMEM_KEEP_MSG  0x00000000  /* messages are not deleted in frontend */
#define ERRMEM_ACK_MSG   0x00000001  /* messages are deleted in frontend */

/* Open a session with the error memory backend
 * The default persistent storage will be assigned
 * automatically to the session. 
 * Paramter: void
 *
 * Return: >= 0: valid session handle to use
 *          < 0: Error
 */
int errmem_backend_open_session(void);

/* Close a session wiht the error memory backend
 * Paramter: sh: session handle received from open call
 *
 * Return:   0: successful completion
 *         < 0: Error
 */
int errmem_backend_close_session(int sh);

/* Specify a persistent storage to use in this session other than the current
 * one. It deletes the current assignment and assigns the requested one.
 * Parameter: sh: session handle received from open call
 *            ps: number of persistent storage (sequence number
 *                in the list of persitent storages from the
 *                command line which was used to start the daemon)
 *                0: only the current assignment will be removed
 *              > 0: number of persistent storage to assign to the session
 *
 * Return:   0: successful completion
 *         < 0: Error
 */
int errmem_backend_set_storage(int sh, int ps);

/* One call reads exactly one message from the assigned persitent storage, 
 * starting with the oldest one up to the latest one.
 * Parameter: sh : session handle received from open call
 *            buf: points to memory area where to store the message. This memory
 *                 has to be provided and freed by the caller.
 * Return:      0: successful completion
 *            < 0: Error
 */
int errmem_backend_read(int sh, struct errmem_message* buf);

/* Erases the content of the specified persistent storage 
 *
 * Parameter: sh: session handle received from open call
 *            ps: number of persistent storage (sequence number
 *                in the list of persitent storages from the
 *                command line which was used to start the daemon)
 *                0: content of all configured persistent storages are erased.
 *              > 0: number of persistent storage which content shall be reased
 *
 * Return:   0: successful completion
 *         < 0: Error
*/
int errmem_backend_erase(int sh, int ps);

/* Instructs the error memory backend (daemon) to dump the current content
 * of the error memory frontend to a non persistent output device like e.g.
 * DLT/CONSOLE/READ.
 * This dump functionality is activated as soon as persistent storages
 * cannot be accessed any longer or no persistent storages are specified.
 * This interface allows to read the current content of the frontend in
 * case the error memory backend cannot make it persistent due to missing
 * accessible persistent storages.
 * See also command line parameters: -g --dlt-output
 * This function allows setting this option at runtime if not already
 * specified by command line parameters.
 * Parameter: sh : session handle received from open call
 *            out: Output interface to dump to:
 *                 ERRMEM_DLT_OUT
 *                 ERRMEM_STD_OUT
 *                 ERRMEM_STD_ERR
 * 
 * Return      0: successful completion
 *           < 0: Error
 */
int errmem_backend_dump_out(int sh, int out);

/* Instructs the backend to dump the content of the persistent storage
 * assigned to the session specified by sh to the output device out 
 * Parameter: sh : session handle received from open call
 *            out: Output interface to dump to:
 *                 ERRMEM_DLT_OUT
 *                 ERRMEM_STD_OUT
 *                 ERRMEM_STD_ERR
 * 
 * Return      0: successful completion
 *           < 0: Error
 */
int errmem_backend_dump(int sh, int out);

/* Instructs the backend to stop reading data from the frontend as well as
 * to stop writing data to persistent storages. When this function returns
 * it is guaranteed that the backend does not do any further actions.
 */
int errmem_backend_stop(int sh);

/* Frees common resources and shall be called when the process using this
 * library will stop execution.
 */
void errmem_backend_deinit(void);
#endif
